/*  RetroArch - A frontend for libretro.
 *  Copyright (C) 2010-2014 - Hans-Kristian Arntzen
 *  Copyright (C) 2011-2015 - Daniel De Matteis
 * 
 *  RetroArch is free software: you can redistribute it and/or modify it under the terms
 *  of the GNU General Public License as published by the Free Software Found-
 *  ation, either version 3 of the License, or (at your option) any later version.
 *
 *  RetroArch is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 *  without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *  PURPOSE.  See the GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along with RetroArch.
 *  If not, see <http://www.gnu.org/licenses/>.
 */


#ifndef __RARCH_NETPLAY_H
#define __RARCH_NETPLAY_H

#include <stdint.h>
#include <stddef.h>
#include <boolean.h>
#include "libretro.h"
#include "libretro_version_1.h"

typedef struct netplay netplay_t;

void input_poll_net(void);

int16_t input_state_net(unsigned port, unsigned device,
      unsigned idx, unsigned id);

void video_frame_net(const void *data, unsigned width,
      unsigned height, size_t pitch);

void audio_sample_net(int16_t left, int16_t right);

size_t audio_sample_batch_net(const int16_t *data, size_t frames);

int16_t input_state_spectate(unsigned port, unsigned device,
      unsigned idx, unsigned id);

int16_t input_state_spectate_client(unsigned port, unsigned device,
      unsigned idx, unsigned id);

/**
 * netplay_new:
 * @server               : IP address of server.
 * @port                 : Port of server.
 * @frames               : Amount of lag frames.
 * @cb                   : Libretro callbacks.
 * @spectate             : If true, enable spectator mode.
 * @nick                 : Nickname of user.
 *
 * Creates a new netplay handle. A NULL host means we're 
 * hosting (user 1).
 *
 * Returns: new netplay handle.
 **/
netplay_t *netplay_new(const char *server,
      uint16_t port, unsigned frames,
      const struct retro_callbacks *cb, bool spectate,
      const char *nick);

/**
 * netplay_free:
 * @netplay              : pointer to netplay object
 *
 * Frees netplay handle.
 **/
void netplay_free(netplay_t *handle);

/**
 * netplay_flip_users:
 * @netplay              : pointer to netplay object
 *
 * On regular netplay, flip who controls user 1 and 2.
 **/
void netplay_flip_users(netplay_t *handle);

/**
 * netplay_pre_frame:   
 * @netplay              : pointer to netplay object
 *
 * Pre-frame for Netplay.
 * Call this before running retro_run().
 **/
void netplay_pre_frame(netplay_t *handle);

/**
 * netplay_post_frame:   
 * @netplay              : pointer to netplay object
 *
 * Post-frame for Netplay.
 * We check if we have new input and replay from recorded input.
 * Call this after running retro_run().
 **/
void netplay_post_frame(netplay_t *handle);

#endif

