/*	$NetBSD$	*/

/*-
 * Copyright (c) 2008 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Julian Coleman.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Board type determination logic from:
 *
 * XFree86: xc/programs/Xserver/hw/xfree86/drivers/sunffb/ffb_driver.c,v 1.11 2002/12/06 02:44:04 tsi Exp 
 *
 * Creator, Creator3D and Elite3D framebuffer driver.
 *
 * Copyright (C) 2000 Jakub Jelinek (jakub@redhat.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * JAKUB JELINEK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <pwd.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/stat.h>

#include "ffbmodes.h"

#define FBCFG_VERSION "0003"
#define FBCFG_VLEN strlen(FBCFG_VERSION)

#define FBNAME "/dev/fb0"
#define SAVEFILE ".ffbreg."

#define DAC_REG 0x0bc06000
#define FBC_REG 0x04000000
#define EXP_ADDR 0x0bc18000

uint32_t	boardtype(const int, char *, const int, const int, const int);
struct ffb_dac_modes *	get_ffb_modes(int *);
void	i2c_dir(int, volatile uint32_t *, volatile uint32_t *);
void	i2c_write(uint8_t, volatile uint32_t *, volatile uint32_t *);
uint8_t	i2c_read(volatile uint32_t *, volatile uint32_t *);
int	i2c_wait_scl(volatile uint32_t *, volatile uint32_t *);
int	i2c_start(volatile uint32_t *, volatile uint32_t *);
void	i2c_stop(volatile uint32_t *, volatile uint32_t *);
int	i2c_byte_write(uint8_t, volatile uint32_t *, volatile uint32_t *);
int	i2c_byte_read(uint8_t *, volatile uint32_t *, volatile uint32_t *);
int	get_edid_data(uint8_t *, volatile uint32_t *, volatile uint32_t *);
int	ask_mode(struct ffb_dac_modes *, const int, char *, const int,
    const uint32_t);
int	save_mode(const char *, struct ffb_dac_modes *, const int, const int);
void	set_ffb_mode(const struct ffb_dac_modes *, volatile uint32_t *,
    volatile uint32_t *, const uint32_t, const int, const int, const int);
int	restore_mode(const char *, volatile uint32_t *, volatile uint32_t *,
    const uint32_t, const int, const int, const int);
int	adjust_display(const char *, struct ffb_dac_modes *,
    volatile uint32_t *, volatile uint32_t *dac_reg_data, const int,
    const int, const int, const int);
double	pclk (const uint32_t);
void	usage(const char *);

uint32_t
boardtype(const int fb, char *fbname,
    const int do_nothing, const int quiet, const int verbose)
{
	volatile uint32_t *fbc_reg, res;
	volatile uint8_t *expansion;
	uint8_t fem, ver, d;
	uint32_t fb_version;

	fbc_reg = mmap(NULL, (size_t) 0x4000, PROT_READ | PROT_WRITE,
	    MAP_PRIVATE, fb, (off_t) FBC_REG);
	if (fbc_reg == MAP_FAILED) {
		fprintf(stderr, "Error mapping FBC: %s\n", strerror(errno));
		return 0;
	}
	expansion = mmap(NULL, (size_t) 0x1, PROT_READ | PROT_WRITE,
	    MAP_PRIVATE, fb, (off_t) EXP_ADDR);
	if (expansion == MAP_FAILED) {
		fprintf(stderr, "Error mapping EXP: %s\n", strerror(errno));
		return 0;
	}
	/* FloatEnableMask (byte offset 0x1540) */
	fem = (uint8_t) fbc_reg[0x550] & 0x7f;
	if (!quiet || verbose) {
		printf("Framebuffer:         %s\n", fbname);
		printf("Board type:          ");
	}
	if (fem == 0x01 || fem == 0x07 || fem == 0x3f) {
		/* AFB is always Z-Buffer and double-buffer */
		fb_version = VERSION_AFB | VERSION_ZBUFFER | \
		    VERSION_DOUBLE_BUFFER;
		if (fem == 0x01 && (!quiet || verbose))
			printf("Elite3D M3 or M6 (without firmware loaded)");
		else if (fem == 0x07 && (!quiet || verbose))
			printf("Elite3D M3 (firmware loaded)\n");
		else if (fem == 0x3f && (!quiet || verbose))
			printf("Elite3D M6 (firmware loaded)\n");
	} else {
		/*
		 * Strapping bits
		 * Need to read these twice, as occasionally we read
		 * a different value first time.
		 */
		ver = expansion[0];
		ver = expansion[0];
		/* Frame Buffer Config 0 (byte offset 0x270) */
		res = fbc_reg[0x9c];
		if (ver & 0x02) {
			if (!quiet || verbose)
				printf("Creator3D ");
			d = 1;
		} else {
			if (!quiet || verbose)
				printf("Creator ");
			d = 0;
		}
		switch (ver & 0x78) {
			case 0x00:
				if (!quiet || verbose)
					printf("(FFB1 prototype)");
				if (d)
					fb_version = VERSION_FFB1_3D;
				else
					fb_version = VERSION_FFB1;
				break;
			case 0x08:
				if (!quiet || verbose)
					printf("(FFB1)");
				if (d)
					fb_version = VERSION_FFB1_3D;
				else
					fb_version = VERSION_FFB1;
				break;
			case 0x18:
				if (!quiet || verbose)
					printf("(FFB1 Speedsort)");
				if (d)
					fb_version = VERSION_FFB1_3D;
				else
					fb_version = VERSION_FFB1;
				break;
			case 0x20:
				if (!quiet || verbose)
					printf("(FFB2/vertical prototype)");
				if (d)
					fb_version = VERSION_FFB2_3D;
				else
					fb_version = VERSION_FFB2;
				break;
			case 0x28:
				if (!quiet || verbose)
					printf("(FFB2/vertical)");
				if (d)
					fb_version = VERSION_FFB2_3D;
				else
					fb_version = VERSION_FFB2;
				break;
			case 0x30:
				if (!quiet || verbose)
					printf("(FFB2+/vertical)");
				if (d)
					fb_version = VERSION_FFB2p3D;
				else
					fb_version = VERSION_FFB2p;
				break;
			case 0x40:
				if (!quiet || verbose)
					printf("(FFB2/horizontal)");
				if (d)
					fb_version = VERSION_FFB2_3D;
				else
					fb_version = VERSION_FFB2;
				break;
			case 0x50:
				if (!quiet || verbose)
					printf("(FFB2+/horizontal)");
				if (d)
					fb_version = VERSION_FFB2p3D;
				else
					fb_version = VERSION_FFB2p;
				break;
			default:
				if (!quiet || verbose)
					printf("(unknown board version)");
				fb_version = 0;
				break;
		}
		if (ver & 0x02)
			fb_version |= VERSION_ZBUFFER;
		if ((ver & 0x01) && ((res & 0x30) != 0x30))
			fb_version |= VERSION_DOUBLE_BUFFER;

		if (!quiet || verbose) {
			if (fb_version & VERSION_ZBUFFER)
				printf(" Z-buffer");
			if (fb_version & VERSION_DOUBLE_BUFFER)
				printf(" double-buffered\n");
			else
				printf(" single-buffered\n");
		}
	}
	return fb_version;
}

#define I2C_DIR_OUT	0
#define I2C_DIR_IN	1
void
i2c_dir(int dir, volatile uint32_t *dac_reg, volatile uint32_t *dac_reg_data)
{
	*dac_reg = REG_MPS;
	if (dir == I2C_DIR_OUT)
		*dac_reg_data = 0x0f;
	else
		*dac_reg_data = 0x08;
	usleep(5);
}

void
i2c_write(uint8_t bits, volatile uint32_t *dac_reg,
    volatile uint32_t *dac_reg_data)
{
/*
printf("i2c_write: %08x\n", bits);
*/
	*dac_reg = REG_MPD;
	*dac_reg_data = bits;
	
}

uint8_t
i2c_read(volatile uint32_t *dac_reg, volatile uint32_t *dac_reg_data)
{
/*
uint32_t res;
*/
	*dac_reg = REG_MPS;
/*
res = *dac_reg_data;
printf("i2c_read: %08x\n", res);
return res & 0xff;
*/
	return (uint8_t) *dac_reg_data && 0xff;
}

#define SCL_TIMEOUT 1000
int
i2c_wait_scl(volatile uint32_t *dac_reg, volatile uint32_t *dac_reg_data)
{
	int timeout = 0;
	uint8_t bits;

	while (timeout < SCL_TIMEOUT) {
		bits = i2c_read(dac_reg, dac_reg_data);
		if (bits & MPD_SCL)
			return 1;
/*
		usleep(1);
*/
	}
	return 0;
return 1;
}

int
i2c_start(volatile uint32_t *dac_reg, volatile uint32_t *dac_reg_data)
{
	i2c_dir(I2C_DIR_OUT, dac_reg, dac_reg_data);
	i2c_write(MPD_SDA | MPD_SCL, dac_reg, dac_reg_data);
/*
	usleep(5);
*/
	i2c_write(MPD_SCL, dac_reg, dac_reg_data);
	if (!i2c_wait_scl(dac_reg, dac_reg_data))
		return 0;
/*
	usleep(4);
*/
	i2c_write(0, dac_reg, dac_reg_data);
	return 1;
}

void
i2c_stop(volatile uint32_t *dac_reg, volatile uint32_t *dac_reg_data)
{
	i2c_dir(I2C_DIR_OUT, dac_reg, dac_reg_data);
	i2c_write(MPD_SCL, dac_reg, dac_reg_data);
/*
	usleep(5);
*/
	i2c_write(MPD_SCL | MPD_SDA, dac_reg, dac_reg_data);
}

int
i2c_byte_write(uint8_t byte, volatile uint32_t *dac_reg,
    volatile uint32_t *dac_reg_data)
{
	uint8_t mask, bit;
	int err = 1;

	i2c_dir(I2C_DIR_OUT, dac_reg, dac_reg_data);
/*
printf("i2c_byte_write: %02x\n", byte);
*/
	for (mask = 0x80; mask > 0; mask >>= 1) {
		bit = byte & mask ? MPD_SDA : 0;
		i2c_write(bit, dac_reg, dac_reg_data);
/*
		usleep(5);
*/
		i2c_write(bit|MPD_SCL, dac_reg, dac_reg_data);
		if (!i2c_wait_scl(dac_reg, dac_reg_data))
			return 0;
/*
		usleep(4);
*/
		i2c_write(bit, dac_reg, dac_reg_data);
	}

	i2c_dir(I2C_DIR_IN, dac_reg, dac_reg_data);
/*
	usleep(5);
*/
	i2c_write(MPD_SDA, dac_reg, dac_reg_data);
/*
	usleep(5);
*/
	i2c_write(MPD_SDA|MPD_SCL, dac_reg, dac_reg_data);
	if (!i2c_wait_scl(dac_reg, dac_reg_data))
		return 0;
	if (i2c_read(dac_reg, dac_reg_data) & MPS_SDA)
		err = 0;
/*
	usleep(4);
*/
	i2c_write(MPD_SDA, dac_reg, dac_reg_data);
	i2c_dir(I2C_DIR_OUT, dac_reg, dac_reg_data);
	i2c_write(0, dac_reg, dac_reg_data);
	return err;
}

int
i2c_byte_read(uint8_t *byte, volatile uint32_t *dac_reg,
    volatile uint32_t *dac_reg_data)
{
	int i;

	i2c_dir(I2C_DIR_IN, dac_reg, dac_reg_data);
	*byte = 0;
	for (i = 0; i < 8; i++) {
		*byte <<= 1;
		i2c_write(MPD_SDA, dac_reg, dac_reg_data);
/*
		usleep(5);
*/
		i2c_write(MPD_SDA|MPD_SCL, dac_reg, dac_reg_data);
		if (!i2c_wait_scl(dac_reg, dac_reg_data))
			return 0;
		if (i2c_read(dac_reg, dac_reg_data) & MPS_SDA)
			*byte |= 1;
/*
		usleep(4);
*/
	}
/*
printf("i2c_byte_read: %02x\n", *byte);
*/
	i2c_write(MPD_SDA, dac_reg, dac_reg_data);
	i2c_dir(I2C_DIR_OUT, dac_reg, dac_reg_data);
	i2c_write(0, dac_reg, dac_reg_data);
/*
	usleep(5);
*/
	i2c_write(MPD_SCL, dac_reg, dac_reg_data);
	if (!i2c_wait_scl(dac_reg, dac_reg_data))
		return 0;
/*
	usleep(4);
*/
	i2c_write(0, dac_reg, dac_reg_data);
	
	return 1;
}

#define MONITOR_I2C_ADDR	0x50
#define DDC_EDID_START		0x00
int
get_edid_data(uint8_t *buf, volatile uint32_t *dac_reg,
    volatile uint32_t *dac_reg_data)
{
	int i;
	uint8_t byte;
	volatile uint32_t val;
	
	*dac_reg = REG_UCT;
	val = *dac_reg_data;
	*dac_reg = REG_UCT;
	*dac_reg_data = val | UCR_BLN;
	
	if (!i2c_start(dac_reg, dac_reg_data))
		goto err;
	if (!i2c_byte_write(MONITOR_I2C_ADDR << 1 | 0, dac_reg, dac_reg_data))
		goto err;
	if (!i2c_byte_write(DDC_EDID_START, dac_reg, dac_reg_data))
		goto err;
	i2c_stop(dac_reg, dac_reg_data);

	if (!i2c_start(dac_reg, dac_reg_data))
		goto err;
	if (!i2c_byte_write(MONITOR_I2C_ADDR << 1 | 1, dac_reg, dac_reg_data))
		goto err;
	for (i = 0; i < EDID_DATA_LEN; i++) {
		if (!i2c_start(dac_reg, dac_reg_data))
			goto err;
		if (!i2c_byte_read(&byte, dac_reg, dac_reg_data))
			goto err;
		buf[i] = byte;
	}

	i2c_stop(dac_reg, dac_reg_data);

	*dac_reg = REG_UCT;
	*dac_reg_data = val;

	return 1;

err:
	i2c_stop(dac_reg, dac_reg_data);

	*dac_reg = REG_UCT;
	*dac_reg_data = val;

	return 0;
}

struct ffb_dac_modes *
get_ffb_modes(int *num_modes)
{
	struct ffb_dac_modes *m;

	m = &dac_mode_list[0];
	*num_modes = sizeof(dac_mode_list) / sizeof(struct ffb_dac_modes);
	return m;
}

int
ask_mode(struct ffb_dac_modes *modes, const int n, char *mbuf,
    const int mbuf_len, const uint32_t fb_version)
{
	int i, j, k, nl;
	struct ffb_dac_modes *m;
	size_t r;

	printf("\nAvailable operations / resolutions:\n");
	printf("  Restore          ");
	printf("  Save             ");
	printf("  Left             ");
	printf("  Right\n");
	printf("  Up               ");
	printf("  Down\n");
	for (i = 0, j = 0, m = modes; i < n; i++, m++) {
		if (m->versions & fb_version) {
			j++;
			printf("  %s", m->modename);
			for (k = strlen(m->modename) + 1; k < mbuf_len; k++)
				printf(" ");
			if (!(j % 4))
				printf("\n");
		}
	}
	if (j % 4)
		printf("\n");
	printf("Enter resolution: ");
	fflush(stdout);
	nl = 0;
	r = read(STDIN_FILENO, mbuf, mbuf_len - 1);
	if (r == -1) {
		fprintf(stderr, "Error reading input: %s\n", strerror(errno));
		return 1;
	}
	while (r-- > 0) {
		if (mbuf[r] == '\n') {
			mbuf[r] = '\0';
			nl = 1;
		}
		if (mbuf[r] >= 'A' && mbuf[r] <= 'Z')
			mbuf[r] = (char) tolower((int) mbuf[r]);
	}
	return nl;
}

int
save_mode(const char *save, struct ffb_dac_modes *saved_mode,
    const int quiet, const int verbose)
{
	int f;
	ssize_t r;

	
	f = open(save, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR);
	if (f == -1) {
		fprintf(stderr, "Error opening save file: %s\n",
		    strerror(errno));
		return 0;
	}
	if (!quiet || verbose)
		printf("Saving current resolution to %s\n", save);
	write(f, saved_mode, sizeof(struct ffb_dac_modes));
	r = close(f);
	if (r == -1) {
		fprintf(stderr, "Error writing save file: %s\n",
			strerror(errno));
		return 0;
	}
	return 1;
}

void
set_ffb_mode(const struct ffb_dac_modes *mode, volatile uint32_t *dac_reg,
    volatile uint32_t *dac_reg_data, const uint32_t fb_version,
    const int do_nothing, const int quiet, const int verbose)
{
	int i;
	uint32_t val;

	if (!quiet || verbose)
		printf("  Disabling video\n");
	/* Disable video */
	*dac_reg = REG_TGC;
	val = *dac_reg_data & 0x7e;
	if (verbose)
		printf("    Register 0x%04x value 0x%04x\n", REG_TGC, val);
	if (!do_nothing) {
		*dac_reg = REG_TGC;
		*dac_reg_data = val;
	}
	/* Disable video and timing generator */

	/*
	 * XXX: disabling the timing generator can cause the machine
	 * to hang, so we only disable the video here. */
	val = val & 0x7e;
	if (verbose)
		printf("    Register 0x%04x value 0x%04x\n", REG_TGC, val);
	if (!do_nothing) {
		*dac_reg = REG_TGC;
		*dac_reg_data = val;
	}

	/* Wait a short time before resetting the video mode */
#define WAIT_TIME 10000
	usleep(WAIT_TIME);

	/* Set horizontal/vertical timing registers */
	if (!quiet || verbose)
		printf("  Setting video mode\n");
	for (i = 0; i < DAC_REG_NUM; i++) {
		val = mode->reg_list[i] & dac_reg_mask[i];
		if (verbose)
			printf("    Register 0x%04x value 0x%04x\n",
			    dac_reg_list[i], val);
		if (!do_nothing) {
			*dac_reg = dac_reg_list[i];
			*dac_reg_data = val;
		}
	}

	/* Enable video */
	if (!quiet || verbose)
		printf("  Enabling video\n");
	val = mode->reg_tgc & dac_reg_tgc_mask;
	if (verbose)
		printf("    Register 0x%04x value 0x%04x\n",
		    REG_TGC, val);
	if (!do_nothing) {
		*dac_reg = REG_TGC;
		*dac_reg_data = val;
	}
}

int
restore_mode(const char *save, volatile uint32_t *dac_reg,
    volatile uint32_t *dac_reg_data, const uint32_t fb_version,
    const int do_nothing, const int quiet, const int verbose)
{
	int f;
	ssize_t r;
	struct ffb_dac_modes saved_mode;

	f = open(save, O_RDONLY, 0);
	if (f == -1) {
		fprintf(stderr, "Error opening save file: %s\n",
		    strerror(errno));
		return 0;
	}
	r = read(f, &saved_mode, sizeof(struct ffb_dac_modes));
	if (r == -1) {
		fprintf(stderr, "Error reading save file: %s\n",
			strerror(errno));
		return 0;
	}
	close(f);

	if (strncmp(FBCFG_VERSION, saved_mode.modename, FBCFG_VLEN)) {
		fprintf(stderr, "Error: save file is for different card version\n");
		return 0;
	}
	if (saved_mode.versions & fb_version) {
		if (!quiet || verbose)
			printf("Restoring saved resolution from %s\n", save);
		set_ffb_mode(&saved_mode, dac_reg, dac_reg_data, fb_version,
		    do_nothing, quiet, verbose);
	} else {
		fprintf(stderr, "Error: unavailable resolution\n");
		return 0;
	}
	return 1;
}

int
adjust_display(const char *dir, struct ffb_dac_modes *saved_mode,
    volatile uint32_t *dac_reg, volatile uint32_t *dac_reg_data,
    const int fb_version, const int quiet, const int verbose,
    const int do_nothing)
{
	int vfp, vbp, hfp, hbp, x;

	vfp = saved_mode->reg_list[REG_NUM_VSS] - 
	    saved_mode->reg_list[REG_NUM_VBS];
	vbp = saved_mode->reg_list[REG_NUM_VBE] -
	     saved_mode->reg_list[REG_NUM_VSE];
	if (saved_mode->reg_list[REG_NUM_PFC] & 0x01)
		x = 4;	/* interleave = 8/2:1 */
	else
		x = 2; /* interleave = 4/2:1 */
	hfp = (saved_mode->reg_list[REG_NUM_HSS] * x) -
	     (saved_mode->reg_list[REG_NUM_HBS] * x);
	hbp = (saved_mode->reg_list[REG_NUM_HBE] * x) -
	     (saved_mode->reg_list[REG_NUM_HSE] * x);

	if (verbose) {
		printf("    Vertical: front porch %d, back porch %d\n",
		    vfp, vbp);
		printf("    Horizontal: front porch %d, back porch %d\n",
		    hfp, hbp);
	}

	/* Horizontal increment is 8 */
	if (!strcmp(dir, "left") || !strcmp(dir, "right")) {
		if (!strcmp(dir, "left")) {
			if (hbp < 16) {
				fprintf(stderr,
				    "Unable to adjust display left\n");
				return 0;
			}
			saved_mode->reg_list[REG_NUM_HBE] -= (8 / x);
			saved_mode->reg_list[REG_NUM_HBS] -= (8 / x);
			saved_mode->reg_list[REG_NUM_HCE] -= (8 / x);
			saved_mode->reg_list[REG_NUM_HCS] -= (8 / x);
		}
		if (!strcmp(dir, "right")) {
			if (hfp < 16) {
				fprintf(stderr,
				    "Unable to adjust display right\n");
				return 0;
			}
			saved_mode->reg_list[REG_NUM_HBE] += (8 / x);
			saved_mode->reg_list[REG_NUM_HBS] += (8 / x);
			saved_mode->reg_list[REG_NUM_HCE] += (8 / x);
			saved_mode->reg_list[REG_NUM_HCS] += (8 / x);
		}
	}

	/* Vertical increment is 1 */
	if (!strcmp(dir, "up") || !strcmp(dir, "down")) {
		if (!strcmp(dir, "up")) {
			if (vbp < 2) {
				fprintf(stderr,
					"Unable to adjust display up\n");
				return 0;
			}
			saved_mode->reg_list[REG_NUM_VBE] -= 1;
			saved_mode->reg_list[REG_NUM_VBS] -= 1;
		}
		if (!strcmp(dir, "down")) {
			if (vfp < 2) {
				fprintf(stderr,
				    "Unable to adjust display down\n");
				return 0;
			}
			saved_mode->reg_list[REG_NUM_VBE] += 1;
			saved_mode->reg_list[REG_NUM_VBS] += 1;
		}
	}
	set_ffb_mode(saved_mode, dac_reg, dac_reg_data, fb_version,
	    do_nothing, quiet, verbose);
	return 0;
}

double
pclk (const uint32_t reg)
{
	int pd;

	pd = (reg >> 11) & 0x07;
	switch (pd) {
		case 0x03:
			pd = 8;
			break;
		case 0x02:
			pd = 4;
			break;
		case 0x01:
			pd = 2;
			break;
		case 0:
		default:
			pd = 1;
			break;
	}
	return 13500000 * (reg & 0x7f) / ((reg >> 7) & 0x0f) / pd;
}

void
usage(const char *name)
{
	printf("Usage:\t%s [-d dev] [-f file] [-n] [-q] [-v] [cmd]\n", name);
	printf("\t%s [-h]\n", name);
}

int
main (int argc, char *argv[])
{
	int do_nothing, quiet, verbose;
	int ch, fb, i, nl;
	int num_ffb_modes;
	int hres, vres, htot, vtot;
	double pc, vref, href;
	static uint32_t fb_version = 0;
	struct passwd *pwd;
	char *c;
	volatile uint32_t *dac_reg, *dac_reg_data;
	struct ffb_dac_modes saved_mode;
	struct ffb_dac_modes *ffb_mode_list, *m;
	char fbname[PATH_MAX] = FBNAME;
	char save_file[PATH_MAX];
	char mbuf[MODENAME_LEN];
	uint8_t edid_info[EDID_DATA_LEN];

	i = 0;
	do_nothing = 0;
	quiet= 0;
	verbose = 0;
	while ((ch = getopt(argc, argv, "d:f:hnqv")) != -1) {
		switch (ch) {
			case 'd':
				strlcpy(fbname, optarg, PATH_MAX - 1);
				break;
			case 'f':
				strlcpy(save_file, optarg, PATH_MAX - 1);
				i = 1;
				break;
			case 'h':
				usage(getprogname());
				return 0;
				break;
			case 'n':
				do_nothing = 1;
				break;
			case 'q':
				quiet = 1;
				break;
			case 'v':
				verbose = 1;
				break;
			case '?':
				return 1;
				break;
		}
	}
	argc -= optind;
	argv += optind;
	if (argc > 0)
		strlcpy(mbuf, argv[0], MODENAME_LEN -1);
	else
		mbuf[0] = '\0';
	if (!i) {
		pwd = getpwuid(geteuid());
		strlcpy(save_file, pwd->pw_dir, PATH_MAX - 1);
		strlcat(save_file, "/", PATH_MAX - 1);
		strlcat(save_file, SAVEFILE, PATH_MAX - 1);
		c = rindex(fbname, '/');
		if (c == NULL)
			strlcat(save_file, fbname, PATH_MAX - 1);
		else {
			c++;
			if (c != '\0')
				strlcat(save_file, c, PATH_MAX - 1);
		}
			
	}
	fb = open(fbname, O_RDWR, 0);
	if (fb == -1) {
		fprintf(stderr, "Error opening FB: %s\n", strerror(errno));
		return 1;
	}

	if (!(fb_version = boardtype(fb, fbname, do_nothing, quiet, verbose)))
		return 1;
	saved_mode.versions = fb_version;

	dac_reg = mmap(NULL, (size_t) 0x4, PROT_READ | PROT_WRITE,
	    MAP_PRIVATE, fb, (off_t) DAC_REG);
	if (dac_reg == MAP_FAILED) {
		fprintf(stderr, "Error mapping DAC: %s\n", strerror(errno));
		return 1;
	}
	dac_reg_data = dac_reg;
	dac_reg_data++;

	for (i = 0; i < DAC_REG_NUM; i++) {
		*dac_reg = dac_reg_list[i];
		saved_mode.reg_list[i] = *dac_reg_data;
	}
	*dac_reg = REG_TGC;
	saved_mode.reg_tgc = *dac_reg_data;
	
	if (saved_mode.reg_list[REG_NUM_PFC] & 0x01)
		i = 4;	/* interleave = 8/2:1 */
	else
		i = 2; /* interleave = 4/2:1 */

	/* XXX: These are wrong in interlaced mode */
	/* XXX: How to check for stereo? */
	hres = (saved_mode.reg_list[REG_NUM_HBS] * i) - \
	    (saved_mode.reg_list[REG_NUM_HBE] * i);
	htot = (saved_mode.reg_list[REG_NUM_HSS] + 1) * i;
	if (saved_mode.reg_tgc & 0x40)
		hres *= 2;	/* Interlaced */

	vres = saved_mode.reg_list[REG_NUM_VBS] - \
	    saved_mode.reg_list[REG_NUM_VBE];
	vtot = saved_mode.reg_list[REG_NUM_VSS] + 1;

	pc = pclk(saved_mode.reg_list[REG_NUM_PLL]);
	href = pc / htot / 1000;	/* kHz */
	vref = pc / (vtot * htot);
	if (!quiet || verbose) {
		printf("Current resolution:  %dx%dx%1.1f%s",
		    hres, vres, vref,
		    saved_mode.reg_tgc & 0x40 ? " interlaced" : "");
		printf(" (%.1lfkHz, %.2lfMHz)", href, pc / 1000000);
		printf(" %svsync %shsync\n",
		    saved_mode.reg_tgc & 0x08 ? "!" : "",
		    saved_mode.reg_tgc & 0x04 ? "!" : "");
	}
	bzero(saved_mode.modename, MODENAME_LEN);
	strcpy(saved_mode.modename, FBCFG_VERSION);
	snprintf(&saved_mode.modename[FBCFG_VLEN], MODENAME_LEN - FBCFG_VLEN,
	    "%dx%dx%1.1f%s", hres, vres, vref,
	    saved_mode.reg_tgc & 0x40 ? " interlaced" : "");

	if (!strcmp(mbuf, "save")) {
		if (!save_mode(save_file, &saved_mode, quiet, verbose))
			return 1;
		else
			return 0;
	}

	ffb_mode_list = get_ffb_modes(&num_ffb_modes);

	if (!get_edid_data(&edid_info[0], dac_reg, dac_reg_data))
		return 1;

	for (i=0; i < EDID_DATA_LEN; i++) {
		printf ("%02x", edid_info[i]);
		if (!((i + 1) % 16))
			printf("\n");
		else if (!((i + 1) % 8))
			printf("  ");
		else
			printf(" ");
	}
	printf("\n");

	nl = 0;
	if (mbuf[0] == '\0')
		nl = ask_mode(ffb_mode_list, num_ffb_modes, &mbuf[0],
		    MODENAME_LEN, fb_version);

	if (!strcmp(mbuf, "restore")) {
		if (!restore_mode(save_file, dac_reg, dac_reg_data, fb_version,
		    do_nothing, quiet, verbose))
			return 1;
		else
			return 0;
	}

	if (!strcmp(mbuf, "left") || !strcmp(mbuf, "right") ||
	    !strcmp(mbuf, "up") || !strcmp(mbuf, "down")) {
		if (!adjust_display(mbuf, &saved_mode, dac_reg, dac_reg_data,
		    fb_version, quiet, verbose, do_nothing))
			return 1;
		else
			return 0;
	}

	for (i = 0, m = ffb_mode_list; i < num_ffb_modes; i++, m++) {
		if (!strcmp(mbuf, m->modename) && (m->versions & fb_version)) {
			if (!quiet || verbose)
				printf("\nSetting resolution to %s\n",
				    mbuf);
			set_ffb_mode(m, dac_reg, dac_reg_data,
			    fb_version, do_nothing, quiet, verbose);
			return 0;
		}
	}
	fprintf(stderr, "Error: unrecognised resolution\n");
	if (mbuf[0] == '\0' && !nl)
		for (i = getchar(); i != '\n'; i = getchar());
	return 1;
}
