/* $Id: packet.c,v 1.4 2008/09/10 13:17:59 agcrooks Exp $ */

/*
 * Copyright (c) 2005 Manuel Freire.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <fcntl.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <openssl/bn.h>
#include <openssl/evp.h>

#include "constants.h"
#include "data.h"
#include "error.h"
#include "bpgutil.h"

#include "packet.h"

/* Packet body reading and writing private functions. */
int pop_raw(BPG_BUF *, void *, size_t);
int pop_uint8(BPG_BUF *, uint8_t *);
int pop_uint16(BPG_BUF *, uint16_t *);
int pop_uint32(BPG_BUF *, uint32_t *);
int pop_mpi(BPG_BUF *, BIGNUM *);
int pop_mpi_encrypted(BPG_BUF *, BIGNUM *, BPG_SECKEY *, BPG_CTX *);
int pop_s2k(BPG_BUF *, S2K *);
int pop_pubkey(BPG_BUF *, BPG_PUBKEY *);
void push_raw(BPG_BUF *, void *, size_t);
void push_uint8(BPG_BUF *, uint8_t);
void push_uint16(BPG_BUF *, uint16_t);
void push_uint32(BPG_BUF *, uint32_t);
void push_mpi(BPG_BUF *, BIGNUM *);
void push_mpi_encrypted(BPG_BUF *, BIGNUM *, S2K *, BPG_CTX *);
void push_s2k(BPG_BUF *, S2K *);

/* Initialize and return a BPG_PKT pointer. */
BPG_PKT *
BPG_PKT_new(void)
{
	BPG_PKT *pkt;
	
	if ((pkt = malloc(sizeof(*pkt))) == NULL) {
		return NULL;
	}
	(void) memset(pkt, 0x0, sizeof(*pkt));

	pkt->body = BPG_BUF_new(0);

	return pkt;
}

/* Free a BPG_PKT pointer. */
void
BPG_PKT_free(BPG_PKT *pkt)
{

	if (pkt != NULL) {
		BPG_BUF_free(pkt->body);
		free(pkt);
	}
}

/* Print a packet in a stream, formatted. */
void
BPG_PKT_print(BPG_PKT *pkt, FILE *fp)
{

	if (pkt != NULL && fp != NULL) {	
		(void)fprintf(fp, "<Packet>\n");
		switch (pkt->format) {
		case PKTFORMAT_OLD:
			(void)fprintf(fp, "\tFormat: Old\n");
			(void)fprintf(fp, "\tLength type: %d\n",
				      pkt->length_type);
			break;
		case PKTFORMAT_NEW:
			(void)fprintf(fp, "\tFormat: New\n");
			break;
		default:
			(void)fprintf(fp, "\tFormat: Unknown\n");
		}
		(void)fprintf(fp, "\tLength: %d\n", pkt->length);
		(void)fprintf(fp, "\tType: %d\n", pkt->type);
		(void)fprintf(fp, "\tBody: ");
		BPG_BUF_print(pkt->body, fp);
		(void)fprintf(fp, "</Packet>\n");
	}
}

/* Read a packet from 'from'. */
int
BPG_PKT_read(BPG_PKT *pkt, unsigned char *from, size_t flen)
{
	int hdrlen;

	if (pkt == NULL || from == NULL || flen < 1) {
		BPG_err(BPGERR_INVALID_ARGUMENT);
		return -1;
	}

	if (flen < 1) {
		BPG_err(BPGERR_PKT_TOO_SHORT);
		return -1;
	}
	if (util_getbit(from[0], 7) == 0) {
		BPG_err(BPGERR_PKT_MALFORMED);
		return -1;
	}
	if (util_getbit(from[0], 6) == 0)
		pkt->format = PKTFORMAT_OLD;
	else
		pkt->format = PKTFORMAT_NEW;

	if (pkt->format == PKTFORMAT_OLD) {
		pkt->type = util_getbitrow(from[0], 5, 2);	
		switch (pkt->type) {
		case PKTTYPE_PKESK:
		case PKTTYPE_SIG:
		case PKTTYPE_SKESK:
		case PKTTYPE_ONEPASS:
		case PKTTYPE_SECKEY:
		case PKTTYPE_PUBKEY:
		case PKTTYPE_SECSUBKEY:
		case PKTTYPE_COMPRESS:
		case PKTTYPE_SYMMENC:
		case PKTTYPE_MARKER:
		case PKTTYPE_LITERAL:
		case PKTTYPE_TRUST:
		case PKTTYPE_USERID:
		case PKTTYPE_PUBSUBKEY:
			break;
		default:
			BPG_err(BPGERR_PKTTYPE_UNKNOWN);
			return -1;
		}
		pkt->length_type = util_getbitrow(from[0], 1, 0);
		switch (pkt->length_type) {
		case PKTLENTYPE_ONE_OCT:
			hdrlen = 2;
			if (flen < hdrlen) {
				BPG_err(BPGERR_PKT_TOO_SHORT);
				return -1;
			}
			pkt->length = from[1];
			break;
		case PKTLENTYPE_TWO_OCT:
			hdrlen = 3;
			if (flen < hdrlen) {
				BPG_err(BPGERR_PKT_TOO_SHORT);
				return -1;
			}
			pkt->length = from[1] << 8 | from[2];
			break;
		case PKTLENTYPE_FOUR_OCT:
			hdrlen = 5;
			if (flen < hdrlen) {
				BPG_err(BPGERR_PKT_TOO_SHORT);
				return -1;
			}
			pkt->length = from[1] << 24 | from[2] << 16
			    | from[3] << 8 | from[4];
			break;
		case PKTLENTYPE_INDETER:
			BPG_err(BPGERR_NOT_IMPLEMENTED);
			return -1;
		default:
			BPG_err(BPGERR_PKTLENTYPE_UNKNOWN);
			return -1;
		}
		pkt->full_length = pkt->length + hdrlen;
		if (pkt->full_length > flen) {
			BPG_err(BPGERR_PKT_MALFORMED);
			return -1;
		}

		BPG_BUF_read(pkt->body, &from[hdrlen], pkt->length);
	}

	if (pkt->format == PKTFORMAT_NEW) {
		BPG_err(BPGERR_PKTFORMAT_NEW_NOT_SUPPORTED);
		return -1;
	}

	return 0;
}

/* Write a packet in a buffer.  Needs having been read or built. */
void
BPG_PKT_write(BPG_PKT *pkt, BPG_BUF *buf)
{
	int hdrlen;
	
	hdrlen = 0; /* quieten lint */
	switch (pkt->format) {
	case PKTFORMAT_OLD:
		switch (pkt->length_type) {
		case PKTLENTYPE_ONE_OCT:
			hdrlen = 2;
			BPG_BUF_resize(buf, hdrlen + pkt->length);
			buf->body[1] = pkt->length;
			break;
		case PKTLENTYPE_TWO_OCT:
			hdrlen = 3;
			BPG_BUF_resize(buf, hdrlen + pkt->length);
			buf->body[1] = pkt->length >> 8 & 0xff;
			buf->body[2] = pkt->length & 0xff;
			break;
		case PKTLENTYPE_FOUR_OCT:
			hdrlen = 5;
			BPG_BUF_resize(buf, hdrlen + pkt->length);
			buf->body[1] = pkt->length >> 24 & 0xff;
			buf->body[2] = pkt->length >> 16 & 0xff; 
			buf->body[3] = pkt->length >> 8 & 0xff; 
			buf->body[4] = pkt->length & 0xff; 
			break;
		case PKTLENTYPE_INDETER:
			BPG_err(BPGERR_NOT_IMPLEMENTED);
			break;
		}
		buf->body[0] = 1 << 7 | pkt->type << 2 | pkt->length_type;
		break;
	case PKTFORMAT_NEW:
		BPG_err(BPGERR_PKTFORMAT_NEW_NOT_SUPPORTED);
		return;
	default:
		BPG_err(BPGERR_PKTFORMAT_UNKNOWN);
		return;
	}
	(void)memcpy(&buf->body[hdrlen], pkt->body->body, pkt->body->len);
}

/* Fill all packet information.  Call after pushing data into the packet. */
void
BPG_PKT_build(BPG_PKT *pkt, uint8_t type, BPG_CTX *opt)
{
	BPG_CTX *ctx;

	ctx = (opt) ? opt : BPG_CTX_new();
	switch (ctx->pktv) {
	case PKTFORMAT_OLD:
		pkt->format = PKTFORMAT_OLD;
		pkt->type = type;
		pkt->length = pkt->body->len;
		if (pkt->length < 256) {
			pkt->length_type = PKTLENTYPE_ONE_OCT;
		} else if (pkt->length < 65536) {
			pkt->length_type = PKTLENTYPE_TWO_OCT;
		} else
			pkt->length_type = PKTLENTYPE_FOUR_OCT;
		break;
	case PKTFORMAT_NEW:
		BPG_err(BPGERR_PKTFORMAT_NEW_NOT_SUPPORTED);
		return;
	default:
		BPG_err(BPGERR_PKTFORMAT_UNKNOWN);
		return;
	}

	if (opt == NULL)
		BPG_CTX_free(ctx);
}

/* Initialize and return a BPG_SIG pointer. */
BPG_SIG *
BPG_SIG_new(void)
{
	BPG_SIG *sig;
	
	if ((sig = malloc(sizeof(*sig))) == NULL) {
		return NULL;
	}
	(void) memset(sig, 0x0, sizeof(*sig));
	sig->dsa_sig = DSA_SIG_new();
	sig->dsa_sig->r = BN_new();
	sig->dsa_sig->s = BN_new();
	return sig;
}

/* Free a BPG_SIG pointer. */
void
BPG_SIG_free(BPG_SIG *sig)
{
	
	if (sig != NULL) {
		DSA_SIG_free(sig->dsa_sig);
		free(sig);
	}
}

/* Print a signature in a stream, formatted. */
void
BPG_SIG_print(BPG_SIG *sig, FILE *fp)
{

	if (sig != NULL && fp != NULL) {
		(void)fprintf(fp, "<Signature>\n");
		(void)fprintf(fp, "\tVersion: %d\n", sig->version);
		(void)fprintf(fp, "\tType: %d\n", sig->type);
		(void)fprintf(fp, "\tPublic Key Algorithm: %d\n",
			      sig->pubkey_algo);
		(void)fprintf(fp, "\tHash Algorithm: %d\n", sig->hash_algo);
		(void)fprintf(fp, "\tFirst two hash octects: %x %x\n",
			      sig->hash_start[0], sig->hash_start[1]);
		switch(sig->version) {
		case SIGVERSION_3:
			(void)fprintf(fp, "\tCreation time: %d\n",
				      sig->creation_time);
			(void)fprintf(fp,
				"\tKey ID: %02x%02x%02x%02x%02x%02x%02x%02x\n",
				sig->key_id[0], sig->key_id[1], sig->key_id[2],
				sig->key_id[3], sig->key_id[4], sig->key_id[5],
				sig->key_id[6], sig->key_id[7]);
			break;
		case SIGVERSION_4:
			(void)fprintf(fp, "\tHashed subpackets length: %d\n",
				      sig->hashed_subpackets_length);
			(void)fprintf(fp, "\tUnhashed subpackets length: %d\n",
				      sig->unhashed_subpackets_length);
			(void)fprintf(fp, "\tHashed subpackets data: ");
			(void)fwrite(sig->hashed_subpackets_data, 1,
				     (unsigned) sig->hashed_subpackets_length, fp);
			(void)fprintf(fp, "\n");
			(void)fprintf(fp, "\tUnhashed subpackets data: ");
			(void)fwrite(sig->unhashed_subpackets_data, 1,
				     (unsigned) sig->unhashed_subpackets_length, fp);
			(void)fprintf(fp, "\n");
			break;
		default:
			break;
		}
		switch (sig->pubkey_algo) {
		case PKA_DSA:
			(void)fprintf(fp, "\t<DSA Signature>\n");
			(void)fprintf(fp, "\t\tr: ");
			fprintf(fp, "%d", BN_num_bytes(sig->dsa_sig->r));
			BN_print_fp(fp, sig->dsa_sig->r);
			(void)fprintf(fp, "\n\t\ts: ");
			BN_print_fp(fp, sig->dsa_sig->s);
			(void)fprintf(fp, "\n\t</DSA Signature>\n");
			break;
		}
		(void)fprintf(fp, "</Signature>\n");
	}
}

/* Read a signature. */
int
BPG_SIG_read(BPG_SIG *sig, BPG_BUF *from)
{
	int err;

	err = 0;
	err |= pop_uint8(from, &sig->version);
	switch (sig->version) {
	case SIGVERSION_3:
		err |= pop_uint8(from, NULL);
		err |= pop_uint8(from, &sig->type);
		err |= pop_uint32(from, &sig->creation_time);
		err |= pop_raw(from, sig->key_id, 8);
		err |= pop_uint8(from, &sig->pubkey_algo);
		err |= pop_uint8(from, &sig->hash_algo);
		err |= pop_raw(from, sig->hash_start, 2);
		switch (sig->pubkey_algo) {
		case PKA_DSA:
			err |= pop_mpi(from, sig->dsa_sig->r);
			err |= pop_mpi(from, sig->dsa_sig->s);
			break;
		default:
			BPG_err(BPGERR_ALGORITHM_NOT_SUPPORTED);
			return -1;
		}
		break;
	case SIGVERSION_4:
		BPG_err(BPGERR_SIGVERSION_4_NOT_SUPPORTED);
		return -1;
	default:
		BPG_err(BPGERR_SIGVERSION_UNKNOWN);
		return -1;
	}

	if (err != 0)
		return -1;
	return 0;
}

int
BPG_SIG_read_packet(BPG_SIG *sig, unsigned char *from, size_t flen)
{
	int ret;
	BPG_PKT *pkt;

	if ((pkt = BPG_PKT_new()) == NULL)
		return -1;
	if (BPG_PKT_read(pkt, from, flen) == -1)
		return -1;
	if (pkt->type != PKTTYPE_SIG) {
		BPG_err(BPGERR_PKTTYPE_UNEXPECTED);
		return -1;
	}

	ret = BPG_SIG_read(sig, pkt->body);

	BPG_PKT_free(pkt);

	return ret;
}

/* Build a signature into a buffer. */
/*ARGSUSED2*/
int
BPG_SIG_build(BPG_SIG *sig, BPG_BUF *to, BPG_CTX *opt)
{

	BPG_BUF_resize(to, 0);
	push_uint8(to, sig->version);
	switch (sig->version) {
	case SIGVERSION_3:
		push_uint8(to, 5);
		push_uint8(to, sig->type);
		push_uint32(to, sig->creation_time);
		push_raw(to, sig->key_id, 8);
		push_uint8(to, sig->pubkey_algo);
		push_uint8(to, sig->hash_algo);
		push_raw(to, sig->hash_start, 2);
		switch (sig->pubkey_algo) {
		case PKA_DSA:
			push_mpi(to, sig->dsa_sig->r);
			push_mpi(to, sig->dsa_sig->s);
			break;
		default:
			BPG_err(BPGERR_ALGORITHM_NOT_SUPPORTED);
			return -1;
		}
		break;
	case SIGVERSION_4:
		BPG_err(BPGERR_SIGVERSION_4_NOT_SUPPORTED);
		break;
	default:
		BPG_err(BPGERR_SIGVERSION_UNKNOWN);
		return -1;
	}
	return 0;
}

/* Build a signature packet into a buffer. */
int
BPG_SIG_build_packet(BPG_SIG *sig, BPG_BUF *to, BPG_CTX *opt)
{
	BPG_PKT *pkt;

	if ((pkt = BPG_PKT_new()) == NULL)
		return -1;

	BPG_SIG_build(sig, pkt->body, opt);
	BPG_PKT_build(pkt, PKTTYPE_SIG, opt);
	BPG_PKT_write(pkt, to);

	BPG_PKT_free(pkt);

	return 0;
}

/* Initialize and return a BPG_SECKEY pointer. */
BPG_SECKEY *
BPG_SECKEY_new(void)
{
	BPG_SECKEY *seckey;
	
	if ((seckey = malloc(sizeof(*seckey))) == NULL) {
		BPG_err(BPGERR_MALLOC);
		return NULL;
	}
	(void) memset(seckey, 0x0, sizeof(*seckey));

	seckey->s2k_usage = 0;
	seckey->symm_algo = 0;
	if ((seckey->s2k = S2K_new()) == NULL) {
		return NULL;
	}
	if ((seckey->pubkey = BPG_PUBKEY_new()) == NULL) {
		return NULL;
	}
	seckey->dsa = seckey->pubkey->dsa;
	seckey->dsa->priv_key = BN_new();

	return seckey;
}

/* Free a BPG_SECKEY pointer. */
void
BPG_SECKEY_free(BPG_SECKEY *seckey)
{

	if (seckey != NULL) {
		BPG_PUBKEY_free(seckey->pubkey);
		S2K_free(seckey->s2k);
		free(seckey);
	}
}

/* Print a secret key in a stream, formatted. */
void
BPG_SECKEY_print(BPG_SECKEY *seckey, FILE *fp)
{
	int i;

	if (seckey != NULL && fp != NULL) {
		(void)fprintf(fp, "<SecretKey>\n");
		(void)fprintf(fp, "\tS2K usage conventions: %d\n",
			      seckey->s2k_usage);
		if (seckey->s2k_usage > 253) { /* FIXME: `== 255' don't work */
			(void)fprintf(fp,
				      "\tSymmetric encryption algorithm: %d\n",
				      seckey->symm_algo);
			(void)fprintf(fp, "\tS2K specifier:\n");
			S2K_print(seckey->s2k, fp);
			(void)fprintf(fp, "\n");
			(void)fprintf(fp, "\tIV: ");
			for (i = 0; i < 8; i++)
				(void)fprintf(fp, "%02x ", seckey->iv[i]);
			(void)fprintf(fp, "\n");
		}
		(void)fprintf(fp, "Checksum: %0d%0d\n", seckey->checksum[0],
			      seckey->checksum[1]);
		BPG_PUBKEY_print(seckey->pubkey, fp);
		(void)fprintf(fp, "</SecretKey>\n");
	}
}

/* Read a secret key. */
int
BPG_SECKEY_read(BPG_SECKEY *seckey, BPG_BUF *from, BPG_CTX *opt)
{
	int err;

	err = 0;
	err |= pop_pubkey(from, seckey->pubkey);
	err |= pop_uint8(from, &seckey->s2k_usage);
	if (seckey->s2k_usage > 253) {	/* FIXME: why `== 255' doesn't work? */
		err |= pop_uint8(from, &seckey->symm_algo);
		err |= pop_s2k(from, seckey->s2k);
	}
	if (seckey->s2k_usage > 0) {		/* encrypted body */
		err |= pop_raw(from, &seckey->iv, 8);
		switch (seckey->pubkey->pubkey_algo) {
		case PKA_DSA:
			err |= pop_mpi_encrypted(from, seckey->dsa->priv_key,
						 seckey, opt);
			break;
		default:
			BPG_err(BPGERR_ALGORITHM_NOT_SUPPORTED);
			return -1;
		}
	} else {				/* unencrypted body */
		switch (seckey->pubkey->pubkey_algo) {
		case PKA_DSA:
			err |= pop_mpi(from, seckey->dsa->priv_key);
			break;
		default:
			BPG_err(BPGERR_ALGORITHM_NOT_SUPPORTED);
			return -1;
		}
	}
	err |= pop_raw(from, &seckey->checksum, 2);

	if (err != 0)
		return -1;
	return 0;
}

int
BPG_SECKEY_read_packet(BPG_SECKEY *seckey, unsigned char *from, size_t flen,
		       BPG_CTX *opt)
{
	int ret;
	BPG_PKT *pkt;

	if ((pkt = BPG_PKT_new()) == NULL)
		return -1;
	if (BPG_PKT_read(pkt, from, flen) == -1)
		return -1;
	if (pkt->type != PKTTYPE_SECKEY) {
		BPG_err(BPGERR_PKTTYPE_UNEXPECTED);
		return -1;
	}

	ret = BPG_SECKEY_read(seckey, pkt->body, opt);

	BPG_PKT_free(pkt);

	return ret;
}

/*ARGSUSED0*/
int
BPG_SECKEY_build(BPG_SECKEY *seckey, BPG_BUF *to, BPG_CTX *opt)
{
	/*
	 * Not yet implemented:
	 *	push_s2k(to, seckey->s2k);
	 *	push_mpi_encrypted(to, seckey->dsa->x);
	 */

	BPG_err(BPGERR_NOT_IMPLEMENTED);
	return -1;

#if 0
	BPG_BUF *buf_pubkey;

	if ((buf_pubkey = BPG_BUF_new(0)) == NULL)
		return -1;

	BPG_PUBKEY_build(seckey->pubkey, buf_pubkey, opt);
	push_raw(to, buf_pubkey->body, buf_pubkey->len);
	push_uint8(to, seckey->s2k_usage);
	if (seckey->s2k_usage > 253) {	/* FIXME: why `== 255' doesn't work? */
		push_uint8(to, seckey->symm_algo);
		/*push_s2k(to, seckey->s2k);*/
	}
	if (seckey->s2k_usage > 0)
		push_raw(to, seckey->iv, 8);
	switch (seckey->pubkey->pubkey_algo) {
	case PKA_DSA:
		/*push_mpi_encrypted(to, seckey->dsa->x);*/
		break;
	default:
		BPG_err(BPGERR_ALGORITHM_NOT_SUPPORTED);
		return -1;
	}
	push_raw(to, seckey->checksum, 2);

	BPG_BUF_free(buf_pubkey);
	return 0;
#endif
}

int
BPG_SECKEY_build_packet(BPG_SECKEY *seckey, BPG_BUF *to, BPG_CTX *opt)
{
	BPG_PKT *pkt;

	if ((pkt = BPG_PKT_new()) == NULL)
		return -1;

	BPG_SECKEY_build(seckey, pkt->body, opt);
	BPG_PKT_build(pkt, PKTTYPE_PUBKEY, opt);
	BPG_PKT_write(pkt, to);

	BPG_PKT_free(pkt);

	return 0;
}

/* Initialize and return a BPG_PUBKEY pointer. */
BPG_PUBKEY *
BPG_PUBKEY_new(void)
{
	BPG_PUBKEY *pubkey;
	
	if ((pubkey = malloc(sizeof(*pubkey))) == NULL) {
		BPG_err(BPGERR_MALLOC);
		return NULL;
	}
	(void) memset(pubkey, 0x0, sizeof(*pubkey));
	pubkey->dsa = DSA_new();
	pubkey->dsa->p = BN_new();
	pubkey->dsa->q = BN_new();
	pubkey->dsa->g = BN_new();
	pubkey->dsa->pub_key = BN_new();
	return pubkey;
}

/* Free a BPG_PUBKEY pointer. */
void
BPG_PUBKEY_free(BPG_PUBKEY *pubkey)
{

	if (pubkey != NULL) {
		DSA_free(pubkey->dsa);
		free(pubkey);
	}
}

/* Print a public key in a stream, formatted. */
void
BPG_PUBKEY_print(BPG_PUBKEY *pubkey, FILE *fp)
{

	if (pubkey != NULL && fp != NULL) {
		(void)fprintf(fp, "<PublicKey>\n");
		(void)fprintf(fp, "\tVersion: %d\n", pubkey->version);
		(void)fprintf(fp, "\tPublic Key Algorithm: %d\n",
			      pubkey->pubkey_algo);
		(void)fprintf(fp, "\tCreation time: %d\n",
			      pubkey->creation_time);
		if (pubkey->version == KEYVERSION_3)
			(void)fprintf(fp,
				      "\tValid days (0=>never expires): %d\n",
				      pubkey->valid_days);
		switch (pubkey->pubkey_algo) {
		case PKA_DSA:
			DSA_print_fp(fp, pubkey->dsa, 0);
			break;
		}
		(void)fprintf(fp, "</PublicKey>\n");
	}
}

/* Read a public key. */
int
BPG_PUBKEY_read(BPG_PUBKEY *pubkey, BPG_BUF *from)
{
	int err;

	err = 0;
	err |= pop_uint8(from, &pubkey->version);
	err |= pop_uint32(from, &pubkey->creation_time);
	if (pubkey->version == KEYVERSION_3)
		err |= pop_uint16(from, &pubkey->valid_days);
	err |= pop_uint8(from, &pubkey->pubkey_algo);
	switch (pubkey->pubkey_algo) {
	case PKA_DSA:
		err |= pop_mpi(from, pubkey->dsa->p);
		err |= pop_mpi(from, pubkey->dsa->q);
		err |= pop_mpi(from, pubkey->dsa->g);
		err |= pop_mpi(from, pubkey->dsa->pub_key);
		break;
	default:
		BPG_err(BPGERR_ALGORITHM_NOT_SUPPORTED);
		return -1;
	}

	if (err != 0)
		return -1;
	return 0;
}

/* Read a public key packet. */
int
BPG_PUBKEY_read_packet(BPG_PUBKEY *pubkey, unsigned char *from, size_t flen)
{
	BPG_PKT *pkt;

	if ((pkt = BPG_PKT_new()) == NULL)
		return -1;
	if (BPG_PKT_read(pkt, from, flen) == -1)
		return -1;
	if (pkt->type != PKTTYPE_PUBKEY) {
		BPG_err(BPGERR_PKTTYPE_UNEXPECTED);
		return -1;
	}

	BPG_PUBKEY_read(pubkey, pkt->body);
	BPG_PKT_free(pkt);

	return 0;
}

/* Build a public key. */
/*ARGSUSED2*/
int
BPG_PUBKEY_build(BPG_PUBKEY *pubkey, BPG_BUF *to, BPG_CTX *opt)
{
	push_uint8(to, pubkey->version);
	push_uint32(to, pubkey->creation_time);
	if (pubkey->version == KEYVERSION_3)
		push_uint16(to, pubkey->valid_days);
	push_uint8(to, pubkey->pubkey_algo);
	switch (pubkey->pubkey_algo) {
	case PKA_DSA:
		push_mpi(to, pubkey->dsa->p);
		push_mpi(to, pubkey->dsa->q);
		push_mpi(to, pubkey->dsa->g);
		push_mpi(to, pubkey->dsa->pub_key);
		break;
	default:
		BPG_err(BPGERR_ALGORITHM_NOT_SUPPORTED);
		return -1;
	}
	return 0;
}

/* Build a public key packet. */
int
BPG_PUBKEY_build_packet(BPG_PUBKEY *pubkey, BPG_BUF *to, BPG_CTX *opt)
{
	BPG_PKT *pkt;

	if ((pkt = BPG_PKT_new()) == NULL)
		return -1;

	BPG_PUBKEY_build(pubkey, pkt->body, opt);
	BPG_PKT_build(pkt, PKTTYPE_PUBKEY, opt);
	BPG_PKT_write(pkt, to);

	BPG_PKT_free(pkt);

	return 0;
}

/* Extract the buffer from a literal data packet. */
/* TODO: consider creating a BPG_LITERAL data type, instead of using BPG_BUF. */
int
BPG_LITERAL_read(BPG_BUF *buf, BPG_BUF *from)
{
	uint32_t	creation_time;
	uint8_t		format;
	uint8_t		len;
	char		name[256 + 1]; /* file name, 1-byte length */
	int		err;

	err = 0;
	err |= pop_uint8(from, &format);
	if (format != BPG_BINARY_FORMAT) {
		BPG_err(BPGERR_NOT_IMPLEMENTED);
		return -1;
	}
	err |= pop_uint8(from, &len);
	err |= pop_raw(from, name, (unsigned)len);
	name[len] = 0;
	err |= pop_uint32(from, &creation_time);

	if (BPG_BUF_resize(buf, from->len - 6 - len) == -1)
		return -1;
	err |= pop_raw(from, buf->body, buf->len);

	return 0;
}

int
BPG_LITERAL_build(BPG_BUF *from, BPG_BUF *to, char *name)
{
	uint8_t len;

	len = (name == NULL) ? 0 : (uint8_t)strlen(name);
	push_uint8(to, BPG_BINARY_FORMAT); /* binary format */
	push_uint8(to, len);
	push_raw(to, name, (unsigned) len);
	push_uint32(to, bpg_time_get());
	push_raw(to, from->body, from->len);
	return 0;
}

/*
 * FIXME: this function is a patch for integrated signing. Rewrite it with
 * a BPG_ONEPASS data type.
 */
int
BPG_ONEPASS_build(BPG_BUF *to, BPG_CTX *opt)
{
	BPG_CTX *ctx;

	ctx = (opt) ? opt : BPG_CTX_new();
	push_uint8(to, (uint8_t)SIGVERSION_3); /* version */
	push_uint8(to, SIGTYPE_BINARY_DOC); /* signature type */
	push_uint8(to, ctx->hash_algo);
	push_uint8(to, ctx->pubkey_algo);
	push_raw(to, ctx->sec_uid, 8);	/* XXX - agc - FIXME - sec uid in signed file??? */
	push_uint8(to, 1);		/* not nested */
	if (opt == NULL)
		BPG_CTX_free(ctx);
	return 0;
}

/* If possible pop next `len' bytes.  If `to' isn't NULL receives popped. */
int
pop_raw(BPG_BUF *buf, void *to, size_t len)
{

	if (len < 1) {
		BPG_err(BPGERR_INVALID_ARGUMENT);
		return -1;
	}
	if (buf->pos + len > buf->len) {
		buf->pos = buf->len;
		BPG_err(BPGERR_PKT_TOO_SHORT);
		return -1;
	}

	if (to != NULL)
		(void)memcpy(to, &buf->body[buf->pos], len);
	buf->pos += len;

	return 0;
}

/* If possible pop next byte.  If `to' is not NULL receives popped data. */
int
pop_uint8(BPG_BUF *buf, uint8_t *to)
{

	if (buf->pos + 1 > buf->len) {
		buf->pos = buf->len;
		BPG_err(BPGERR_PKT_TOO_SHORT);
		return -1;
	}

	if (to != NULL)
		*to = (uint8_t)buf->body[buf->pos];
	buf->pos += 1;

	return 0;
}

/* If possible pop next 2 bytes.  If `to' is not NULL receives popped data. */
int
pop_uint16(BPG_BUF *buf, uint16_t *to)
{

	if (buf->pos + 2 > buf->len) {
		buf->pos = buf->len;
		BPG_err(BPGERR_PKT_TOO_SHORT);
		return -1;
	}

	if (to != NULL)
		*to = buf->body[buf->pos] << 8 | buf->body[buf->pos + 1];
	buf->pos += 2;

	return 0;
}

/* If possible pop next 4 bytes.  If `to' is not NULL receives popped data. */
int
pop_uint32(BPG_BUF *buf, uint32_t *to)
{

	if (buf->pos + 4 > buf->len) {
		buf->pos = buf->len;
		BPG_err(BPGERR_PKT_TOO_SHORT);
		return -1;
	}

	if (to != NULL)
		*to = buf->body[buf->pos] << 24 | buf->body[buf->pos + 1] << 16
		    | buf->body[buf->pos + 2] << 8 | buf->body[buf->pos + 3];
	buf->pos += 4;

	return 0;
}

/* If possible pop next public key. If `to' isn't NULL receives data. */
int
pop_mpi(BPG_BUF *buf, BIGNUM *to)
{
	int len;

	if (buf->pos + 2 > buf->len) {
		buf->pos = buf->len;
		BPG_err(BPGERR_PKT_TOO_SHORT);
		return -1;
	}

	len = ((buf->body[buf->pos] << 8 | buf->body[buf->pos + 1]) + 7) / 8;
	if (buf->pos + 2 + len > buf->len) {
		buf->pos = buf->len;
		BPG_err(BPGERR_PKT_TOO_SHORT);
		return -1;
	}

	if (to != NULL)
		to = BN_bin2bn(&(buf->body[buf->pos + 2]), len, to);
	buf->pos += 2 + len;

	return 0;
}

/* If possible pop next encrypted MPI. If `to' isn't NULL receives data. */
int
pop_mpi_encrypted(BPG_BUF *buf, BIGNUM *to, BPG_SECKEY *seckey, BPG_CTX *opt)
{
	const EVP_CIPHER	*ctype;
	EVP_CIPHER_CTX		 cctx;
	BPG_CTX			*ctx = NULL;
	BPG_BUF			*decrypted = NULL;
	BPG_BUF			*key = NULL;
	int			 len;
	int			 ret;

	ret = -1;

	if ((decrypted = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	if ((key = BPG_BUF_new(0)) == NULL) {
		goto close_and_ret;
	}
	ctx = (opt) ? opt : BPG_CTX_new();

	/* TODO: check size is enough */
	/* TODO: check seckey */

	/* Decrypt */
	if (ctx->pass == NULL)
		ctx->pass = (char *)getpass("Secret key passphrase:");
	S2K_getkey(seckey->s2k, ctx->pass, seckey->symm_algo, key);
	switch (seckey->symm_algo) {
	case SKA_CAST5:
		ctype = (const EVP_CIPHER *)EVP_cast5_cfb();
		break;
	default:
		BPG_err(BPGERR_ALGORITHM_NOT_SUPPORTED);
		goto close_and_ret;
	}
	EVP_CIPHER_CTX_init(&cctx);
	EVP_DecryptInit_ex(&cctx, ctype, NULL, key->body, seckey->iv);
	BPG_BUF_resize(decrypted,
			(unsigned) 2 + EVP_CIPHER_CTX_block_size(&cctx));
	EVP_DecryptUpdate(&cctx, decrypted->body, (int *)&decrypted->len,
			  &buf->body[buf->pos], 2);
	len = ((decrypted->body[0] << 8 | decrypted->body[1]) + 7) / 8;
	BPG_BUF_resize(decrypted,
			(unsigned) len + EVP_CIPHER_CTX_block_size(&cctx));
	EVP_DecryptUpdate(&cctx, decrypted->body, (int *)&decrypted->len,
			  &buf->body[buf->pos + 2], len);
	EVP_CIPHER_CTX_cleanup(&cctx);
	if (to != NULL)
		to = BN_bin2bn(decrypted->body, (int)decrypted->len, to);	
	buf->pos += decrypted->len;

	ret = 0;

close_and_ret:
	if (opt == NULL) BPG_CTX_free(ctx);
	if (decrypted) BPG_BUF_free(decrypted);
	if (key) BPG_BUF_free(key);

	return ret;
}

int
pop_s2k(BPG_BUF *buf, S2K *s2k)
{
	S2K	*tmps2k = NULL;
	int	 len;
	int	 ret;

	ret = -1;

	tmps2k = (s2k) ? s2k : S2K_new();
	len = S2K_read(tmps2k, &buf->body[buf->pos], buf->len - buf->pos);
	if (len == -1) {
		goto close_and_ret;
	}
	buf->pos += len;

	ret = 0;

close_and_ret:
	if (s2k == NULL) S2K_free(tmps2k);
	return ret;
}

int
pop_pubkey(BPG_BUF *buf, BPG_PUBKEY *pubkey)
{
	BPG_PUBKEY	*tmpkey = NULL;
	int		 ret;

	ret = 0;
	tmpkey = (pubkey) ? pubkey : BPG_PUBKEY_new();
	if (BPG_PUBKEY_read(tmpkey, buf) == -1) {
		ret = -1;
	}
	if (pubkey == NULL)
		BPG_PUBKEY_free(tmpkey);
	return ret;
}

/* Push `len' bytes from `from' at the end of the buffer. */
void
push_raw(BPG_BUF *buf, void *from, size_t len)
{

	if (len < 1)
		return;
	if (BPG_BUF_resize(buf, buf->len + len) == -1)
		return;		/* TODO: this resize can be optimized */

	if (from != NULL)
		(void)memcpy(&buf->body[buf->len - len], from, len);
}

/* Push a byte at the end of the buffer. */
void
push_uint8(BPG_BUF *buf, uint8_t val)
{

	if (BPG_BUF_resize(buf, buf->len + 1) == -1)
		return;		/* TODO: this resize can be optimized */

	buf->body[buf->len - 1] = val;
}

/* Push an uint16 at the end of the buffer (big-endian). */
void
push_uint16(BPG_BUF *buf, uint16_t val)
{

	if (BPG_BUF_resize(buf, buf->len + 2) == -1)
		return;		/* TODO: this resize can be optimized */

	util_uint16_to_str(val, &buf->body[buf->len - 2]);
}

/* Push an uint32 at the end of the buffer (big-endian). */
void
push_uint32(BPG_BUF *buf, uint32_t val)
{

	if (BPG_BUF_resize(buf, buf->len + 4) == -1)
		return;		/* TODO: this resize can be optimized */

	util_uint32_to_str(val, &buf->body[buf->len - 4]);
}

/* Push a multi precision integer at the end of the buffer. */
void
push_mpi(BPG_BUF *buf, BIGNUM *bn)
{
	int bnlen;
	unsigned char *ptr;

	bnlen = BN_num_bytes(bn);
	if (BPG_BUF_resize(buf, buf->len + 2 + bnlen) == -1)
		return;		/* TODO: this resize can be optimized */
	ptr = &buf->body[buf->len - 2 - bnlen];

	/* MPI = len in bits (big-endian) + BIGNUM */
	/*LINTED*/
	/* XXX - agc - not sure how this works */
	ptr[0] = (bnlen * 8) >> 8;
	ptr[1] = (bnlen * 8) & 0xff;
	BN_bn2bin(bn, &ptr[2]);
}

/*ARGSUSED0*/
void
push_mpi_encrypted(BPG_BUF *buf, BIGNUM *bn, S2K *s2k, BPG_CTX *opt)
{
}
