#!/usr/bin/awk -f
#	$NetBSD: mkregtable.awk,v 1.1 2014/07/16 20:59:58 riastradh Exp $
#
# Copyright (c) 2014 The NetBSD Foundation, Inc.
# All rights reserved.
#
# This code is derived from software contributed to The NetBSD Foundation
# by Taylor R. Campbell.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
# ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
# TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
# PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
# BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

function howmany(x, n) {
	return int((x + (n - 1)) / n)
}

function hi(x, n) {
	return int(x / 2^n)
}

function lo(x, n) {
	return (x % 2^n)
}

function rcsid(s) {
	sub("^\\$", "", s)
	sub("\\$$", "", s)

	return s;
}

BEGIN {
	state = "INITIAL"
}

state == "INITIAL" {
	gpu = $1
	maxreg = lastreg = $2
	state = "REGS"
	noffsets = 0
}

state == "REGS" &&
$1 ~ /0x[0-9a-fA-F]/ &&
$2 ~ /[_a-zA-Z0-9]*/ {
	if (!seen[$1]) {
		seen[$1] = 1
		offset[noffsets++] = $1
		if ($1 > maxreg)
			maxreg = $1
	}
}

END {
	# We do this in 16-bit arithmetic to avoid overflow in case
	# this ever runs on a system whose awk uses single-precision
	# floats or 32-bit integers or something horrible like that.
	nentries = howmany(hi(maxreg, 2), 32)
	for (i = 0; i < nentries; i++) {
		# No hex numeric literals in awk!
		table_hi[i] = 2^16 - 1
		table_lo[i] = 2^16 - 1
	}
	for (o = 0; o < noffsets; o++) {
		reg = hi(offset[o], 2)
		wi = hi(reg, 5)
		bi = lo(reg, 5)
		# Clear the bit.  No bitwise operations, but if the
		# offsets don't overlap, the bit is guaranteed to be
		# set so we can just subtract 2^n.
		if (hi(bi, 4) != 0)
			table_hi[wi] -= 2^lo(bi, 4)
		else
			table_lo[wi] -= 2^bi
	}
	printf("/*\n")
	printf(" *\t%c%s%c\n", "$", "NetBSD", "$")
	printf(" *\n")
	printf(" * Stand back!  This file was automagically generated by\n")
	printf(" *\t%s\n", rcsid("$NetBSD: mkregtable.awk,v 1.1 2014/07/16 20:59:58 riastradh Exp $"))
	printf(" */\n")
	printf("\n")
	printf("static const uint32_t %s_reg_safe_bm[%u] = {\n", gpu, nentries)
	for (i = 0; i < nentries; i++) {
		if ((i % 4) == 0)
			printf("\t")
		printf("0x%04X%04X,", table_hi[i], table_lo[i])
		if (((i % 4) == 3) || (i == (nentries - 1)))
			printf("\n")
		else
			printf(" ")
	}
	printf("};\n")
}
