# -*- coding: utf-8 -*-
from setuptools import setup

setup(
    name='pipx',
    version='%%PORTVERSION%%',
    description='Install and Run Python Applications in Isolated Environments',
    long_description='<p align="center">\n<a href="https://pypa.github.io/pipx/">\n<img align="center" src="https://github.com/pypa/pipx/raw/main/logo.png" width="200"/>\n</a>\n</p>\n\n# pipx — Install and Run Python Applications in Isolated Environments\n\n<p align="center">\n<a href="https://github.com/pypa/pipx/raw/main/pipx_demo.gif">\n<img src="https://github.com/pypa/pipx/raw/main/pipx_demo.gif"/>\n</a>\n</p>\n\n<p align="center">\n<a href="https://github.com/pypa/pipx/actions">\n<img src="https://github.com/pypa/pipx/workflows/tests/badge.svg?branch=main" alt="image" /></a> <a href="https://badge.fury.io/py/pipx"><img src="https://badge.fury.io/py/pipx.svg" alt="PyPI version"></a> <a href="https://badge.fury.io/py/pipx"><img src="https://pepy.tech/badge/pipx"></a>\n\n</p>\n\n**Documentation**: https://pypa.github.io/pipx/\n\n**Source Code**: https://github.com/pypa/pipx\n\n_For comparison to other tools including pipsi, see [Comparison to Other Tools](https://pypa.github.io/pipx/comparisons/)._\n\n## Install pipx\n\n### On macOS\n\n```\nbrew install pipx\npipx ensurepath\n```\n\nUpgrade pipx with `brew update && brew upgrade pipx`.\n\n### On Linux, install via pip (requires pip 19.0 or later)\n\n```\npython3 -m pip install --user pipx\npython3 -m pipx ensurepath\n```\n\nUpgrade pipx with `python3 -m pip install --user -U pipx`.\n\n### On Windows, install via pip (requires pip 19.0 or later)\n\n```\n# If you installed python using the app-store, replace `python` with `python3` in the next line.\npython -m pip install --user pipx\n```\n\nIt is possible (even most likely) the above finishes with a WARNING looking similar to this:\n\n```\nWARNING: The script pipx.exe is installed in `<USER folder>\\AppData\\Roaming\\Python\\Python3x\\Scripts` which is not on PATH\n```\n\nIf so, go to the mentioned folder, allowing you to run the pipx executable directly.\nEnter the following line (even if you did not get the warning):\n\n```\npipx ensurepath\n```\n\nThis will add both the above mentioned path and the `%USERPROFILE%\\.local\\bin` folder to your search path.\nRestart your terminal session and verify `pipx` does run.\n\nUpgrade pipx with `python3 -m pip install --user -U pipx`.\n\n### Shell completions\n\nShell completions are available by following the instructions printed with this command:\n\n```\npipx completions\n```\n\nFor more details, see the [installation instructions](https://pypa.github.io/pipx/installation/).\n\n## Overview: What is `pipx`?\n\npipx is a tool to help you install and run end-user applications written in Python. It\'s roughly similar to macOS\'s `brew`, JavaScript\'s [npx](https://medium.com/@maybekatz/introducing-npx-an-npm-package-runner-55f7d4bd282b), and Linux\'s `apt`.\n\nIt\'s closely related to pip. In fact, it uses pip, but is focused on installing and managing Python packages that can be run from the command line directly as applications.\n\n### How is it Different from pip?\n\npip is a general-purpose package installer for both libraries and apps with no environment isolation. pipx is made specifically for application installation, as it adds isolation yet still makes the apps available in your shell: pipx creates an isolated environment for each application and its associated packages.\n\npipx does not ship with pip, but installing it is often an important part of bootstrapping your system.\n\n### Where Does `pipx` Install Apps From?\n\nBy default, pipx uses the same package index as pip, [PyPI](https://pypi.org/). pipx can also install from all other sources pip can, such as a local directory, wheel, git url, etc.\n\nPython and PyPI allow developers to distribute code with "console script entry points". These entry points let users call into Python code from the command line, effectively acting like standalone applications.\n\npipx is a tool to install and run any of these thousands of application-containing packages in a safe, convenient, and reliable way. **In a way, it turns Python Package Index (PyPI) into a big app store for Python applications.** Not all Python packages have entry points, but many do.\n\nIf you would like to make your package compatible with pipx, all you need to do is add a [console scripts](https://python-packaging.readthedocs.io/en/latest/command-line-scripts.html#the-console-scripts-entry-point) entry point. If you\'re a poetry user, use [these instructions](https://python-poetry.org/docs/pyproject/#scripts).\n\n## Features\n\n`pipx` enables you to\n\n- Expose CLI entrypoints of packages ("apps") installed to isolated environments with the `install` command. This guarantees no dependency conflicts and clean uninstalls!\n- Easily list, upgrade, and uninstall packages that were installed with pipx\n- Run the latest version of a Python application in a temporary environment with the `run` command\n\nBest of all, pipx runs with regular user permissions, never calling `sudo pip install` (you aren\'t doing that, are you? 😄).\n\n### Walkthrough: Installing a Package and its Applications With `pipx`\n\nYou can globally install an application by running\n\n```\npipx install PACKAGE\n```\n\nThis automatically creates a virtual environment, installs the package, and adds the package\'s associated applications (entry points) to a location on your `PATH`. For example, `pipx install pycowsay` makes the `pycowsay` command available globally, but sandboxes the pycowsay package in its own virtual environment. **pipx never needs to run as sudo to do this.**\n\nExample:\n\n```\n>> pipx install pycowsay\n  installed package pycowsay 2.0.3, Python 3.7.3\n  These apps are now globally available\n    - pycowsay\ndone! ✨ 🌟 ✨\n\n\n>> pipx list\nvenvs are in /home/user/.local/pipx/venvs\napps are exposed on your $PATH at /home/user/.local/bin\n   package pycowsay 2.0.3, Python 3.7.3\n    - pycowsay\n\n\n# Now you can run pycowsay from anywhere\n>> pycowsay mooo\n  ____\n< mooo >\n  ====\n         \\\n          \\\n            ^__^\n            (oo)\\_______\n            (__)\\       )\\/\\\n                ||----w |\n                ||     ||\n\n```\n\n### Installing from Source Control\n\nYou can also install from a git repository. Here, `black` is used as an example.\n\n```\npipx install git+https://github.com/psf/black.git\npipx install git+https://github.com/psf/black.git@branch  # branch of your choice\npipx install git+https://github.com/psf/black.git@ce14fa8b497bae2b50ec48b3bd7022573a59cdb1  # git hash\npipx install https://github.com/psf/black/archive/18.9b0.zip  # install a release\n```\n\n### Walkthrough: Running an Application in a Temporary Virtual Environment\n\nThis is an alternative to `pipx install`.\n\n`pipx run` downloads and runs the above mentioned Python "apps" in a one-time, temporary environment, leaving your system untouched afterwards.\n\nThis can be handy when you need to run the latest version of an app, but don\'t necessarily want it installed on your computer.\n\nYou may want to do this when you are initializing a new project and want to set up the right directory structure, when you want to view the help text of an application, or if you simply want to run an app in a one-off case and leave your system untouched afterwards.\n\nFor example, the blog post [How to set up a perfect Python project](https://sourcery.ai/blog/python-best-practices/) uses `pipx run` to kickstart a new project with [cookiecutter](https://github.com/cookiecutter/cookiecutter), a tool that creates projects from project templates.\n\nA nice side benefit is that you don\'t have to remember to upgrade the app since `pipx run` will automatically run a recent version for you.\n\nOkay, let\'s see what this looks like in practice!\n\n```\npipx run APP [ARGS...]\n```\n\nThis will install the package in an isolated, temporary directory and invoke the app. Give it a try:\n\n```\n> pipx run pycowsay moo\n\n  ---\n< moo >\n  ---\n   \\   ^__^\n    \\  (oo)\\_______\n       (__)\\       )\\/\\\n           ||----w |\n           ||     ||\n\n\n```\n\nNotice that you **don\'t need to execute any install commands to run the app**.\n\nAny arguments after the application name will be passed directly to the application:\n\n```\n> pipx run pycowsay these arguments are all passed to pycowsay!\n\n  -------------------------------------------\n< these arguments are all passed to pycowsay! >\n  -------------------------------------------\n   \\   ^__^\n    \\  (oo)\\_______\n       (__)\\       )\\/\\\n           ||----w |\n           ||     ||\n\n```\n\nRe-running the same app is quick because pipx caches Virtual Environments on a per-app basis. The caches only last a few days, and when they expire, pipx will again use the latest version of the package. This way you can be sure you\'re always running a new version of the package without having to manually upgrade.\n\nIf the app name does not match that package name, you can use the `--spec` argument to specify the package to install and app to run separately:\n\n```\npipx run --spec PACKAGE APP\n```\n\nYou can also specify specific versions, version ranges, or extras:\n\n```\npipx run APP==1.0.0\n```\n\n### Running from Source Control\n\nYou can also run from a git repository. Here, `black` is used as an example.\n\n```\npipx run --spec git+https://github.com/psf/black.git black\npipx run --spec git+https://github.com/psf/black.git@branch black  # branch of your choice\npipx run --spec git+https://github.com/psf/black.git@ce14fa8b497bae2b50ec48b3bd7022573a59cdb1 black  # git hash\npipx run --spec https://github.com/psf/black/archive/18.9b0.zip black # install a release\n```\n\n### Running from URL\n\nYou can run .py files directly, too.\n\n```\npipx run https://gist.githubusercontent.com/cs01/fa721a17a326e551ede048c5088f9e0f/raw/6bdfbb6e9c1132b1c38fdd2f195d4a24c540c324/pipx-demo.py\npipx is working!\n```\n\n### Summary\n\nThat\'s it! Those are the most important commands `pipx` offers. To see all of pipx\'s documentation, run `pipx --help` or see the [docs](https://pypa.github.io/pipx/docs/).\n\n## Testimonials\n\n<div>\n"Thanks for improving the workflow that pipsi has covered in the past. Nicely done!"\n<div style="text-align: right; margin-right: 10%; font-style:italic;">\n—<a href="https://twitter.com/jezdez">Jannis Leidel</a>, PSF fellow, former pip and Django core developer, and founder of the Python Packaging Authority (PyPA)\n</div>\n</div>\n\n<p></p>\n\n<div>\n"My setup pieces together pyenv, poetry, and pipx. [...] For the things I need, it’s perfect."\n<div style="text-align: right; margin-right: 10%; font-style:italic;">\n—<a href="https://twitter.com/jacobian">Jacob Kaplan-Moss</a>, co-creator of Django in blog post <a href="https://jacobian.org/2019/nov/11/python-environment-2020/">My Python Development Environment, 2020 Edition</a>\n</div>\n</div>\n\n<p></p>\n\n<div>\n"I\'m a big fan of pipx. I think pipx is super cool."\n<div style="text-align: right; margin-right: 10%; font-style:italic;">\n—<a href="https://twitter.com/mkennedy">Michael Kennedy</a>, co-host of PythonBytes podcast in <a href="https://pythonbytes.fm/episodes/transcript/139/f-yes-for-the-f-strings">episode 139</a>\n</div>\n</div>\n\n<p></p>\n\n## Credits\n\npipx was inspired by [pipsi](https://github.com/mitsuhiko/pipsi) and [npx](https://github.com/npm/npx). It was created by [Chad Smith](https://github.com/cs01/) and has had lots of help from [contributors](https://github.com/pypa/pipx/graphs/contributors). The logo was created by [@IrishMorales](https://github.com/IrishMorales).\n\npipx is maintained by a team of volunteers (in alphabetical order)\n\n- [Bernát Gábor](https://github.com/gaborbernat)\n- [Chad Smith](https://github.com/cs01) - co-lead maintainer\n- [Matthew Clapp](https://github.com/itsayellow) - co-lead maintainer\n- [Tzu-ping Chung](https://github.com/uranusjr)\n\n## Contributing\n\nIssues and Pull Requests are definitely welcome! Check out [Contributing](https://pypa.github.io/pipx/contributing/) to get started.\nEveryone who interacts with the pipx project via codebase, issue tracker, chat rooms, or otherwise is expected to follow\nthe [PSF Code of Conduct](https://github.com/pypa/.github/blob/main/CODE_OF_CONDUCT.md).\n',
    author_email='Chad Smith <chadsmith.software@gmail.com>',
    classifiers=[
        'License :: OSI Approved :: MIT License',
        'Operating System :: OS Independent',
        'Programming Language :: Python',
        'Programming Language :: Python :: 3 :: Only',
        'Programming Language :: Python :: 3.7',
        'Programming Language :: Python :: 3.8',
        'Programming Language :: Python :: 3.9',
        'Programming Language :: Python :: 3.10',
    ],
    install_requires=[
        'argcomplete>=1.9.4',
        'colorama>=0.4.4; sys_platform == "win32"',
        'importlib-metadata>=3.3.0; python_version < "3.8"',
        'packaging>=20.0',
        'userpath>=1.6.0',
    ],
    entry_points={
        'console_scripts': [
            'pipx = pipx.main:cli',
        ],
    },
    packages=[
        'pipx',
        'pipx.commands',
    ],
    package_dir={
        '': 'src',
    },
)
